//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.1
import QtGraphicalEffects 1.0

import wesual.Controls 1.0

FocusScope {
    id : comboBox

    property list<ComboBoxItem> items
    property int currentIndex : -1
    property bool editable : false
    property alias validator : editableText.validator

    property bool subtle : false

    readonly property bool open : popup.popupVisible

    readonly property string currentText : {
        if (currentIndex >= 0 && currentIndex < items.length) {
            return items[currentIndex].text;
        } else if (currentIndex === -1) {
            return editableText.text;
        } else {
            return "";
        }
    }

    signal itemChosen(int index)

    function openPopup()  {
        popup.activeIndex = currentIndex;
        popup.show();
        popup.popupItem.focus = true;
    }

    onOpenChanged : {
        if (!open)
            comboBox.forceActiveFocus();
    }

    // Key handling
    Keys.onSpacePressed : {
        if (!open) {
            comboBox.openPopup();
        }
    }
    Keys.onDownPressed : {
        if (!open) {
            comboBox.openPopup();
        }
    }

    width : 100
    implicitHeight : 26
    z : popup.popupVisible ? 10000 : 0
    activeFocusOnTab : true

    onActiveFocusChanged : {
        if (!activeFocus && editable) {
            if(!editableText.acceptableInput)
                editableText.undoTextInput();
        }
    }

    Rectangle {
        id : background

        z : 1
        anchors.fill : parent
        color : UI.color(UI.PrimaryBase)

        Behavior on color {
            ColorAnimation { duration : 0 }
        }
        border {
            width : 1
            color : {
                if (subtle) {
                    if (!comboBox.enabled)
                        return UI.color(UI.DisabledBorder);

                    if (mouseArea.pressed || comboBox.open) {
                        return UI.color(UI.ControlBorderPress);
                    } else if (mouseArea.containsMouse || comboBox.activeFocus) {
                        return UI.color(UI.ControlBorderHover);
                    }

                    return "transparent"
                } else {
                    if (!comboBox.enabled)
                        return UI.color(UI.DisabledBorder);

                    if (mouseArea.pressed || comboBox.open) {
                        return UI.color(UI.ControlBorderPress);
                    } else if (mouseArea.containsMouse || comboBox.activeFocus) {
                        return UI.color(UI.ControlBorderHover);
                    }
                    return UI.color(UI.PrimaryControlBorder);
                }
            }
        }

        states : [
            State {
                name : "disabled"
                when : !comboBox.enabled

                changes : [
                    PropertyChanges {
                        target : background
                        color  : UI.color(UI.PrimaryBase)
                    },
                    PropertyChanges {
                        target : indicator
                        color  : subtle ?
                                     UI.color(UI.PrimaryBase) :
                                     UI.color(UI.DisabledButton)
                    },
                    PropertyChanges {
                        target : textShown
                        color  : UI.color(UI.DisabledText)
                    }

                ]
            },
            State {
                name : "selected"
                when : mouseArea.pressed || comboBox.open

                changes : [
                    PropertyChanges {
                        target : background
                        color : subtle ?
                                    UI.color(UI.PrimaryBase) :
                                    UI.color(UI.PrimaryPress)
                    },
                    PropertyChanges {
                        target : indicator
                        color  : UI.color(UI.PrimaryBase)
                    },
                    PropertyChanges {
                        target : textShown
                        color  : subtle ?
                                     UI.color(UI.PrimaryTextColor) :
                                     UI.color(UI.SelectedTextColor)
                    }
                ]
            },
            State {
                name : "hovered"
                when : mouseArea.containsMouse || comboBox.activeFocus

                PropertyChanges {
                    target : background
                    color  : subtle ?
                                 UI.color(UI.PrimaryBase) :
                                 UI.color(UI.PrimaryHover)
                }
            }
        ]

        Behavior on border.color {
            ColorAnimation { duration : 0 }
        }
        Text {
            id : textShown
            anchors {
                fill : parent
                leftMargin  : 6
                rightMargin : indicator.width
            }
            text  : comboBox.currentText
            elide : Text.ElideRight
            font {
                family : UI.fontFamily(UI.PrimaryFont)
                weight : UI.fontWeight(UI.PrimaryFont)
                pixelSize : 14
            }
            verticalAlignment : Qt.AlignVCenter
            visible : !editable
        }
        TextField {
            id : editableText

            anchors.left : parent.left
            anchors.leftMargin : 1
            anchors.verticalCenter: parent.verticalCenter
            width : parent.width - indicator.width - 10
            height : parent.height - 2
            border.width : 0
            text : textShown.text
            font {
                family : UI.fontFamily(UI.PrimaryFont)
                weight : UI.fontWeight(UI.PrimaryFont)
                pixelSize : 14
            }
            verticalAlignment : Qt.AlignVCenter
            visible : editable
            onEditFinished : {
                currentIndex = -1;
            }
        }
        ColorizedImage {
            id : indicator
            anchors {
                right : parent.right
                rightMargin : 6
                verticalCenter : parent.verticalCenter
                verticalCenterOffset : 1
            }
            source : "qrc:/controls/icons/dropdown-indicator.png"
            color : subtle && !mouseArea.containsMouse ?
                        UI.color(UI.PrimaryBase) :
                        UI.color(UI.DropdownIndicatorIcon)
        }
    }

    MouseArea {
        id : mouseArea

        z : 2
        hoverEnabled : true
        anchors.right : !editable ? undefined : parent.right
        width  : !editable ? parent.width : parent.width - editableText.width
        height : parent.height
        acceptedButtons : Qt.LeftButton
        onPressed : comboBox.openPopup()
        enabled : comboBox.enabled
    }

    ComboBoxPopup {
        id : popup

        z : 0

        anchors.top : parent.bottom

        items : comboBox.items
        comboBox : comboBox
        currentSelection : comboBox.currentIndex
        onAccepted : {
            comboBox.currentIndex = result;
            editableText.text = items[result].text;
            comboBox.itemChosen(result);
            popup.hide();
        }
    }
}
